// ---------------------------------------------------------------------------
// - Wedge.hpp                                                               -
// - afnix:geo service - image wedge class definition                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_WEDGE_HPP
#define  AFNIX_WEDGE_HPP

#ifndef  AFNIX_IMAGE_HPP
#include "Image.hpp"
#endif

#ifndef  AFNIX_PIXEL_HPP
#include "Pixel.hpp"
#endif

namespace afnix {
  
  /// The Wedge class is a single or multiple band image class which
  /// aggregates a slice of pixels. The wedge is characterized by its
  /// depth. The wedge format is defined by the pixel format which is
  /// the same across the slices.
  /// @author amaury darsch

  class Wedge : public Image {
  protected:
    /// the pixel format
    Pixel::t_pfmt d_pfmt;
    /// the wedge depth
    long d_dpth;
    
  public:
    /// create a default wedge
    Wedge (void);

    /// copy construct this wedge
    /// @param that the wedge to copy
    Wedge (const Wedge& that);

    /// copy move this wedge
    /// @param that the wedge to move
    Wedge (Wedge&& that) noexcept;

    /// assign an wedge to this one
    /// @param that the wedge to assign
    Wedge& operator = (const Wedge& that);

    /// move an wedge to this one
    /// @param that the wedge to move
    Wedge& operator = (Wedge&& that) noexcept;

    // serialize this slice
    /// @param os the output stream
    void wrstream (OutputStream& os) const;

    /// deserialize this slice
    /// @param is the input stream
    void rdstream (InputStream& os);
    
    /// reset this wedge
    void reset (void);
    
    /// @return the pixel format
    virtual Pixel::t_pfmt getpfmt (void) const;
    
    /// @return the wedge depth
    virtual long getdpth (void) const;
    
    /// set an wedge component by position and pixel
    /// @param x the wedge x position
    /// @param y the wedge y position
    /// @param d the wedge d position
    /// @param pixl the pixel value
    virtual void setpixl (const long x, const long y, const long d,
			  const Pixel& pixl) =0;

    /// @return a pixel by position
    virtual Pixel getpixl (const long x, const long y, const long d) const =0;
    
  public:
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
