/* $Id: er_json.c,v 1.8 2013-02-22 15:20:04 cgarcia Exp $
 *
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2001-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-02-22 15:20:04 $
 * $Revision: 1.8 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <er_json.h>


/**
 * @defgroup JSON utilities
 *
 * This module allows to write some objects in JSON format
 *
 * @par Synopsis:
 * @code
 *   #include <er_json.h>
 * @endcode
 */

/**@{*/

/**
 * @brief
 *   Write a JSON file with the contents of a frameset.
 *
 * @param frameset  The frameset to export
 * @param json_file Name of the JSON file to write 
 *
 * @return @c CPL_ERROR_NONE on success.
 *
 * This function will export the content of a frameset to a JSON format
 * which can be machine readable. The format looks like:
 * [
 *  {
 *    "name": "ifu_trace.fits",
 *    "category": "IFU_TRACE"
 *  },
 *  {
 *    "name": "ifu_transmission.fits",
 *    "category": "IFU_TRANSMISSION"
 *  }
 * ]
 * 
 */
cpl_error_code er_frameset_to_json(const cpl_frameset * frameset, 
                                   const char* json_file)
{
    FILE             * fjson;
    cpl_size           nframes;
    cpl_size           iframe;

    /* Sanity checks  */ 
    if(!frameset || !json_file)
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                 "Null input");

    /* Open the output file */
    fjson = fopen(json_file, "w");
    if(!fjson)
    {
        errno = 0;
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_CREATED,
                "Cannot open JSON file for writing");
    }
    
    /* Open a JSON list */
    fprintf(fjson, "[\n");
    
    /* Loop on frames */
    nframes = cpl_frameset_get_size(frameset);
    iframe = 0;

    for(iframe = 0 ; iframe < nframes; iframe++)
    {
        const cpl_frame  * frame;
        char             * filename;
        char             * category;
        
        /* Get this frame */
        frame = cpl_frameset_get_position_const(frameset, iframe);
        filename = er_json_escape_string(cpl_frame_get_filename(frame));
        category = er_json_escape_string(cpl_frame_get_tag(frame));
        
        /* Open a JSON item */
        fprintf(fjson, "  {\n");
        
        /* Write this frame fields */
        if(strlen(filename) > 0)
            fprintf(fjson, "    \"name\": \"%s\",\n",filename);
        if(strlen(category) > 0)
            fprintf(fjson, "    \"category\": \"%s\"\n",category);
        
        /* Close JSON item */
        if(iframe == cpl_frameset_get_size(frameset) - 1)
            fprintf(fjson, "  }\n");
        else
            fprintf(fjson, "  },\n");
        cpl_free(filename);
        cpl_free(category);
    }
    /* Close JSON list */
    fprintf(fjson, "]\n");
    fclose(fjson);
    
    return CPL_ERROR_NONE;
}

/**
 * @brief
 *   Write a text file with the contents of a frameset.
 *
 * @param frameset  The frameset to export
 * @param text_file Name of the JSON file to write 
 *
 * @return @c CPL_ERROR_NONE on success.
 *
 * This function will export the contents of the frameset to a text human
 * readable file, like the one used for the input sof. 
 * The output will look like this:
 * ifu_trace.fits     IFU_TRACE
 * ifu_transmission.fits     IFU_TRANSMISSION
 */
cpl_error_code er_frameset_to_text(const cpl_frameset * frameset, 
                                   const char* text_file)
{
    FILE             * ftext;
    cpl_size           nframes;
    cpl_size           iframe;

    /* Sanity checks  */ 
    if(!frameset || !text_file)
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                 "Null input");

    /* Open the output file */
    ftext = fopen(text_file, "w");
    if(!ftext)
    {
        errno = 0;
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_CREATED,
                "Cannot open file %s for writing", text_file);
    }
    
    /* Loop on frames */
    nframes = cpl_frameset_get_size(frameset);
    iframe = 0;

    for(iframe = 0 ; iframe < nframes; iframe++)
    {
        const cpl_frame  * frame;
        const char       * filename;
        const char       * category;

        /* Get this frame */
        frame = cpl_frameset_get_position_const(frameset, iframe);
        filename = cpl_frame_get_filename(frame);
        category = cpl_frame_get_tag(frame);
        
        /* Writing a frame*/
        fprintf(ftext, "%s     %s", filename, category);
        if(iframe != cpl_frameset_get_size(frameset) - 1)
            fprintf(ftext, "\n");
        
        iframe++;
    }
    fclose(ftext);
    
    return CPL_ERROR_NONE;
}

/**
 * @brief
 *   Write a JSON file with the contents of a parameterlist.
 *
 * @param plist       The parameter list  to export
 * @param recipe_name The recipe this parameter list applies to
 * @param json_file   Name of the JSON file to write 
 *
 * @return @c CPL_ERROR_NONE on success.
 *
 * This function exports the content of a parameter list to a machine readable
 * JSON format. The output looks like:
 *[
 * {
 *   "name": "vimos.Parameters.stacking.singleframes",
 *   "value": true,
 *   "display_name": "AllowSingleFrames",
 *   "description": "Frame combination method is ignored.",
 *   "recipe": "vmbias"
 * },
 * {
 *   "name": "vimos.Parameters.stacking.method",
 *   "value": "Median",
 *   "display_name": "StackMethod",
 *   "description": "Frame combination method",
 *   "recipe": "vmbias"
 * }
 *]
 */
cpl_error_code er_recipe_parameterlist_to_json(const cpl_parameterlist * plist,
                                               const char * recipe_name,
                                               const char * json_file)
{
    FILE                * fjson;
    const cpl_parameter * param;

    /* Sanity checks  */ 
    if(!plist || !recipe_name || !json_file)
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                 "Null input");

    /* Open the output file */
    fjson = fopen(json_file, "w");
    if(!fjson)
    {
        errno = 0;
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_CREATED,
                "Cannot open JSON file for writing");
    }
    
    /* Open a JSON list */
    fprintf(fjson, "[\n");
    
    /* Loop on parameters */
    param = cpl_parameterlist_get_first_const(plist);

    while (param != NULL)
    {
        char              * description;
        char              * name;
        char              * display_name;
        cpl_type            valtype;
        cpl_parameter_class partype;
        int                 ivalue;
        int                 bvalue;
        double              dvalue;
        char              * svalue;
        int                 enum_size;
        int                 ienum;
        
        description = er_json_escape_string(cpl_parameter_get_help(param));
        name = er_json_escape_string(cpl_parameter_get_name(param));
        display_name = er_json_escape_string(
            cpl_parameter_get_alias(param, CPL_PARAMETER_MODE_CLI));
        if(display_name == NULL)
        {
            cpl_error_reset();
            display_name = er_json_escape_string(cpl_parameter_get_name(param));
        }
        valtype = cpl_parameter_get_type(param);
        partype = cpl_parameter_get_class(param);
        
        /* Open a JSON item */
        fprintf(fjson, "  {\n");
        
        /* Write this parameter fields */
        if(strlen(name) > 0)
            fprintf(fjson, "    \"name\": \"%s\",\n",name);
        /* Write value and value type */
        switch (valtype)
        {
        case CPL_TYPE_BOOL:
            bvalue = cpl_parameter_get_bool(param);
            if (bvalue == 0)
                fprintf(fjson, "    \"value\": false,\n");
            else
                fprintf(fjson, "    \"value\": true,\n");
                fprintf(fjson, "    \"valtype\": \"bool\",\n");
            break;

        case CPL_TYPE_INT:
            ivalue = cpl_parameter_get_int(param);
            fprintf(fjson, "    \"value\": %d,\n",ivalue);
            fprintf(fjson, "    \"valtype\": \"int\",\n");
            break;

        case CPL_TYPE_DOUBLE:
            dvalue = cpl_parameter_get_double(param);
            fprintf(fjson, "    \"value\": %.17g,\n",dvalue); //IEEE 754 double
            fprintf(fjson, "    \"valtype\": \"double\",\n");
            break;

        case CPL_TYPE_STRING:
            svalue = er_json_escape_string(cpl_parameter_get_string(param));
            fprintf(fjson, "    \"value\": \"%s\",\n",svalue);
            fprintf(fjson, "    \"valtype\": \"string\",\n");
            cpl_free(svalue);
            break;

        default:
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                     "Type not supported for cpl_propertylist");
            break;
        }

        /* Write parameter type (called class inside CPL) */
        switch (partype)
        {
        case CPL_PARAMETER_CLASS_VALUE:
            fprintf(fjson, "    \"partype\": \"value\",\n");
            break;

        /* If a range, write the minimum and maximum.
           Only integers and doubles are supported */
        case CPL_PARAMETER_CLASS_RANGE:
            fprintf(fjson, "    \"partype\": \"range\",\n");
            switch (valtype)
            {
            case CPL_TYPE_INT:
                ivalue = cpl_parameter_get_range_min_int(param);
                fprintf(fjson, "    \"valmin\": %d,\n",ivalue);
                ivalue = cpl_parameter_get_range_max_int(param);
                fprintf(fjson, "    \"valmax\": %d,\n",ivalue);
                break;

            case CPL_TYPE_DOUBLE:
                dvalue = cpl_parameter_get_range_min_double(param);
                fprintf(fjson, "    \"valmin\": %.17g,\n",dvalue);
                dvalue = cpl_parameter_get_range_max_double(param);
                fprintf(fjson, "    \"valmax\": %.17g,\n",dvalue);
                break;
            default:
                return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                         "Range parameters not supported for this type");
                break;
            }
            break;
        /* If an enum, write the list of alternatives as a JSON list.
           Only integers, doubles and strings are supported */
        case CPL_PARAMETER_CLASS_ENUM:
            fprintf(fjson, "    \"partype\": \"enum\",\n");
            enum_size = cpl_parameter_get_enum_size(param);
            fprintf(fjson, "    \"valenum\": [");
            for(ienum = 0; ienum < enum_size; ienum++)
            {
                switch (valtype)
                {
                case CPL_TYPE_INT:
                    ivalue = cpl_parameter_get_enum_int(param, ienum);
                    fprintf(fjson, " %d ",ivalue);
                    break;
   
                case CPL_TYPE_DOUBLE:
                    dvalue = cpl_parameter_get_enum_double(param, ienum);
                    fprintf(fjson, " %.17g ",dvalue);
                    break;
   
                case CPL_TYPE_STRING:
                    svalue = er_json_escape_string(
                        cpl_parameter_get_enum_string(param, ienum));
                    fprintf(fjson, " \"%s\" ",svalue);
                    cpl_free(svalue);
                    break;
                default:
                    return cpl_error_set_message(cpl_func, 
                             CPL_ERROR_TYPE_MISMATCH,
                             "Enum parameters not supported for this type");
                    break;
                 }
                 if(ienum != enum_size - 1)
                    fprintf(fjson, ",");
            }
            fprintf(fjson, "],\n");
            break;
        default:
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                     "Parameter class not supported for cpl_propertylist");
            break;
        }
        if(strlen(display_name) > 0)
            fprintf(fjson, "    \"display_name\": \"%s\",\n",display_name);
        if(strlen(description) > 0)
        {
            if(strlen(recipe_name) > 0) //Description is not the last field
                fprintf(fjson, "    \"description\": \"%s\",\n",description);
            else                        //Description is the last field
                fprintf(fjson, "    \"description\": \"%s\"\n",description);
        }
        if(strlen(recipe_name) > 0)
            fprintf(fjson, "    \"recipe\": \"%s\"\n",recipe_name);
        
        /* Close JSON item */
        if(param == cpl_parameterlist_get_last_const(plist))
            fprintf(fjson, "  }\n");
        else
            fprintf(fjson, "  },\n");
        param = cpl_parameterlist_get_next_const(plist);

        cpl_free(description);
        cpl_free(name);
        cpl_free(display_name);
    }
    /* Close JSON list */
    fprintf(fjson, "]\n");
    fclose(fjson);
    
    return CPL_ERROR_NONE;
}

char * er_json_escape_string(const char * str)
{
    char * escaped_str;
    size_t i, j;
    if(str == NULL)
        return NULL;

    escaped_str = cpl_malloc(2*strlen(str) * sizeof(char)+1);
    for(i = 0, j = 0; i < strlen(str); i++)
    {
        if(str[i] == 0x22 || str[i] == 0x5C || str[i] == 0x2F )
        {
            escaped_str[j++] = '\\';
            escaped_str[j++] = str[i];
        }
        else if(str[i] == 0x08)
        {
            escaped_str[j++] = '\\';
            escaped_str[j++] = 'b';
        }
        else if(str[i] == 0x09)
        {
            escaped_str[j++] = '\\';
            escaped_str[j++] = 't';
        }
        else if(str[i] == 0x0A)
        {
            escaped_str[j++] = '\\';
            escaped_str[j++] = 'n';
        }
        else if(str[i] == 0x0C)
        {
            escaped_str[j++] = '\\';
            escaped_str[j++] = 'f';
        }
        else if(str[i] == 0x0D)
        {
            escaped_str[j++] = '\\';
            escaped_str[j++] = 'r';
        }
        else
            escaped_str[j++] = str[i];
    }
    escaped_str[j++]='\0';    
    escaped_str = cpl_realloc(escaped_str, j);
    
    return escaped_str;
}

/**@}*/
