/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <limits.h>
#include <values.h>
#include "exrinput.h"
#include "exroutput.h"

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    exrinput_t *exrin;
    double *scanline;
    int width, height;
    double maxv;
    int y;

    if( argc < 3 ) {
        fprintf( stderr, "Usage: exrnormalize input.exr output.exr [maxval]\n" );
        return 1;
    }

    exrin = exrinput_new( argv[ 1 ] );
    if( !exrin ) {
        fprintf( stderr, "exrnormalize: Cannot open %s.\n", argv[ 1 ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );
    fprintf( stdout, "exrnormalize: %s [%dx%d]\n", argv[ 1 ], width, height );

    scanline = malloc( width * 4 * sizeof( double ) );
    if( !scanline ) {
        fprintf( stderr, "exrnormalize: Cannot allocate memory.\n" );
        exrinput_delete( exrin );
        return 1;
    }

    exrout = exroutput_new( argv[ 2 ], width, height );
    if( !exrout ) {
        fprintf( stderr, "exrnormalize: Cannot open %s.\n", argv[ 2 ] );
        exrinput_delete( exrin );
        return 1;
    }

    if( argc > 3 ) {
        maxv = atof( argv[ 3 ] );
        fprintf( stderr, "\nexrnormalize: Setting maximum value: %.3f\n", maxv );
    } else {
        maxv = -DBL_MAX;

        for( y = 0; y < height; y++ ) {
            double *inscanline = exrinput_get_scanline( exrin, y );
            int x;

            fprintf( stderr, "exrnormalize: Scanning %d\r", y );

            for( x = 0; x < width; x++ ) {
                double r = inscanline[ (x * 4) + 0 ];
                double g = inscanline[ (x * 4) + 1 ];
                double b = inscanline[ (x * 4) + 2 ];

                if( r > maxv ) maxv = r;
                if( g > maxv ) maxv = g;
                if( b > maxv ) maxv = b;
            }
        }
        fprintf( stderr, "\nexrnormalize: Maximum value: %.3f\n", maxv );
    }

    for( y = 0; y < height; y++ ) {
        double *inscanline = exrinput_get_scanline( exrin, y );
        int x;

        fprintf( stderr, "exrnormalize: Writing %d\r", y );

        for( x = 0; x < width; x++ ) {
            scanline[ (x * 4) + 0 ] = inscanline[ (x * 4) + 0 ] / maxv;
            scanline[ (x * 4) + 1 ] = inscanline[ (x * 4) + 1 ] / maxv;
            scanline[ (x * 4) + 2 ] = inscanline[ (x * 4) + 2 ] / maxv;
            scanline[ (x * 4) + 3 ] = inscanline[ (x * 4) + 3 ];
        }
        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\nexrnormalize: Done.\n" );
    exroutput_delete( exrout );
    exrinput_delete( exrin );
    free( scanline );
    return 0;
}

