#
# Parser for a subset of the iCalendar data format (RFC 2445).
#


from cal.Date import Date
from cal.Calendar import Calendar
from cal.CalEvent import CalEvent
from cal.Recurrences import Recurrences
from cal.RRule import RRule


_WEEK = ["MO", "TU", "WE", "TH", "FR", "SA", "SU"]



#
# Case-insenisitive string matching.
#
def _match(a, b):

    return (a.upper() == b.upper())



def _parse_contentline(line):

    index = line.find(":")
    
    left = line[:index]
    value = line[index + 1:]
    
    index2 = left.find(";")
    if (index2 != -1):
        name = left[:index2]
        params = left[index2 + 1:]
    else:
        name = left
        params = ""
        
    return (name, _parse_params(params), value)



def _parse_params(params):

    dict = {}
    for p in params.split(";"):
        if (p):
            name, value = p.split("=")
            dict[name.upper()] = value
    #end for

    return dict



def _parse_stringlist(values):

    ret = []
    parts = values.split(",")
    for p in parts:
        ret.append(p.strip())

    return ret



def _parse_intlist(values):

    ret = []
    parts = _parse_stringlist(values)
    for p in parts:
        ret.append(int(p))

    return ret




#
# Parses a time value.
#
def _parse_time(value):

    parts = value.split("T")
    is_utc = (value[-1] == "Z")

    year = int(parts[0][:4])
    month = int(parts[0][4:6])
    day = int(parts[0][6:8])

    if (len(parts) > 1):
        hours = int(parts[1][:2])
        mins = int(parts[1][2:4])
        secs = int(parts[1][4:6])
    else:
        hours = 0
        mins = 0
        secs = 0

    return Date(year, month, day, hours, mins, secs, is_utc)



#
# Parses a list of weekdays.
#
def _parse_weekdays(values):

    weekdays = []
    parts = _parse_stringlist(values)
    for p in parts:
        index = _WEEK.index(p[-2:])
        if (len(p) > 2):
            week = int(p[:-2])
        else:
            week = 0

        if (week >= 0):
            weekdays.append(10 * week + index)
        else:
            weekdays.append(10 * week - index)
    #end for

    return weekdays





#
# Parser for loading data in the iCalendar format (RFC 2445).
#
class iCalLoader:

    def __init__(self):

        # stack of created objects
        self.__objects = []



    #
    # Opens and parses the given iCalendar file.
    #
    def load(self, filename):

        fd = open(filename, "r")
        data = fd.read()
        fd.close()
        
        return self.__parse(data)




    def __parse(self, data):

        # unwrap long lines
        data = data.replace("\r\n ", "")
        data = data.replace("\n ", "")

        data = data.replace("\,", ",")

        lines = data.splitlines()
        lines.reverse()

        while (1):
            l = lines.pop()
            name, params, value = _parse_contentline(l)
            
            if (_match(name, "BEGIN") and _match(value, "VCALENDAR")):
                self.__parse_vcalendar(lines)
                break
        #end while

        return self.__objects[-1]



    #
    # Parses calendar data.
    #
    def __parse_vcalendar(self, lines):

        calendar = Calendar()
        self.__objects.append(calendar)

        while (1):
            l = lines.pop()
            name, params, value = _parse_contentline(l)

            if (_match(name, "END") and _match(value, "VCALENDAR")): break

            elif (_match(name, "BEGIN") and _match(value, "VEVENT")):
                self.__parse_vevent(lines)

            elif (_match(name, "VERSION")):
                pass

            elif (_match(name, "PRODID")):
                pass
        #end while

        #self.__objects.pop()



    #
    # Parses event data.
    #
    def __parse_vevent(self, lines):

        event = CalEvent()
        self.__objects[-1].add_event(event)
        self.__objects.append(event)

        rrule = event.get_recurrences()
        event.set_recurrences(rrule)
        
        while (1):
            l = lines.pop()
            name, params, value = _parse_contentline(l)
            tzid = params.get("TZID", "")

            if (_match(name, "END") and _match(value, "VEVENT")): break

            elif (_match(name, "DTSTART")):
                date = _parse_time(value)
                event.set_start(date)

            elif (_match(name, "DTEND")):
                date = _parse_time(value)
                event.set_end(date)
            

            elif (_match(name, "CATEGORIES")):
                categs = value.split(",")
                event.set_categories(categs)

            elif (_match(name, "SUMMARY")):
                event.set_summary(value)

            elif (_match(name, "LOCATION")):
                event.set_location(value)

            elif (_match(name, "RDATE")):
                date = _parse_time(value)
                rrule.add_date(date)

            elif (_match(name, "EXDATE")):
                date = _parse_time(value)
                rrule.add_exdate(date)

            elif (_match(name, "RRULE")):
                values = _parse_params(value)
                rrule.add_rule(self.__parse_rrule(values))

        #end while

        self.__objects.pop()





    #
    # Parses a recurrency rule.
    #
    def __parse_rrule(self, values):

        rule = RRule()

        # [bysetpos, bymonths, byweeknrs, byyeardays, bymonthdays, bydays]
        bysets = [[]] * 6

        for key, value in values.items():
            if (_match(key, "FREQ")):
                if (_match(value, "YEARLY")):
                    rule.set_delta(RRule.YEAR, 1)
                    rule.set_frequency(RRule.YEAR)
                elif (_match(value, "MONTHLY")):
                    rule.set_delta(RRule.MONTH, 1)
                    rule.set_frequency(RRule.MONTH)
                elif (_match(value, "WEEKLY")):
                    rule.set_delta(RRule.DAY, 7)
                    rule.set_frequency(RRule.WEEK)
                elif (_match(value, "DAILY")):
                    rule.set_delta(RRule.DAY, 1)
                    rule.set_frequency(RRule.DAY)

            elif (_match(key, "INTERVAL")): rule.set_interval(int(value))
            elif (_match(key, "COUNT")): rule.set_count(int(value))
            elif (_match(key, "UNTIL")):
                date = _parse_time(value)
                rule.set_until(date)

            elif (_match(key, "BYDAY")):
                weekdays = _parse_weekdays(value)
                rule.set_by(RRule.DAY, weekdays)

            elif (_match(key, "BYMONTHDAY")):
                rule.set_by(RRule.MONTHDAY, _parse_intlist(value))

            elif (_match(key, "BYYEARDAY")):
                rule.set_by(RRule.YEARDAY, _parse_intlist(value))

            elif (_match(key, "BYWEEKNO")):
                rule.set_by(RRule.WEEKNO, _parse_intlist(value))

            elif (_match(key, "BYMONTH")):
                rule.set_by(RRule.MONTH, _parse_intlist(value))

            elif (_match(key, "BYSETPOS")):
                rule.set_by(RRule.SETPOS, _parse_intlist(value))

        #end for

        return rule
