-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/gui_preferences.adb,v $
--  Description     : Preference GUI                                         --
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2003/11/16 12:11:18 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------

with Ada.Strings.Unbounded;    use Ada.Strings.Unbounded;
with Ada.Text_Io;              use Ada.Text_IO;
with Ada.Exceptions;           use Ada.Exceptions;
with Ada.Strings;              use Ada.Strings;
with Ada.Strings.Fixed;        use Ada.Strings.Fixed;

with Glib;                     use Glib;
with Gtk.Box;                  use Gtk.Box;
with Gtk.Button;               use Gtk.Button;
with Gtk.Check_Button;         use Gtk.Check_Button;
with Gtk.Combo;                use Gtk.Combo;
with Gtk.GEntry;               use Gtk.GEntry;
with Gtk.Enums;                use Gtk.Enums;
with Gtk.Frame;                use Gtk.Frame;
with Gtk.Handlers;             use Gtk.Handlers;
with Gtk.Label;                use Gtk.Label;
with Gtk;                      use Gtk;

with Project_File;             use Project_File;
with Preferences;              use Preferences;

package body GUI_Preferences is

   Version : constant String := "$Id: gui_preferences.adb,v 1.13 2003/11/16 12:11:18 merdmann Exp $";

   -- ********************************************************************** --
   -- ***                    L O C A L   D A T A                         *** --
   -- ********************************************************************** --

   type Preference_Record is record
         Name        : Unbounded_String := Null_Unbounded_String;
         Description : Unbounded_String := Null_Unbounded_String;
         Value_Entry : Gtk_Entry;
      end record;

   PT          : array( 1..50 ) of Preference_Record;
   Pref_Prefix : constant String := "preference";

   -- ********************************************************************** --
   -- ***               L O C A L   P R O C E D U R E S                  *** --
   -- ********************************************************************** --

   -------------------
   -- Get_Next_Free --
   -------------------
   function Get_Next_Free(
      Name : in String ) return Natural is
   begin
      for I in PT'Range loop
         if PT(I).Name = Null_Unbounded_String then
            PT(I).Name := To_Unbounded_String( Name );
            return I;
         end if;
      end loop;

      raise No_More_Preferences;
   end Get_Next_Free;

   ---------------------
   -- Project_Default --
   ---------------------
   function Project_Default(
      Name      : in String;
      Default   : in String ) return String is
      -- return the actual default value according to the following
      -- rules:
      --
      --   R.1 - If there is a section "prefrences.<name>" in the project
      --         file return this value.
      --   R.2 - If there is no such entry create one with default
      --         value passed in the argument.
      --
   begin
      if not Section_Exists( Pref_Prefix, Name ) then
         Add( Pref_Prefix, Name, Default );
      end if;

      return Project_File.Get( Pref_Prefix, Name );
   end Project_Default;

   -----------
   -- Place --
   -----------
   procedure Place(
      VBox        : in Gtk_Box;
      Name        : in String;
      Description : in String;
      Default     : in String  := "";
      Visible     : in Boolean := True;
      Maxlength   : in Natural := 30 ) is
      ---
      Descr       : Gtk_Label;
      Field       : Gtk_Entry;
      Hbox        : Gtk_Box;
      Next_Free   : Natural := Get_Next_Free(Name);
   begin
      Gtk_New_Hbox (Hbox);
      Set_Border_Width( Hbox, 4 );
      Show_All( Hbox );

      Gtk_New( Descr, Description & " : " );
      Pack_Start(Hbox, Descr, Expand => False );

      Gtk_New( Field );

      Set_Text( Field, Project_Default(Name, Default) );
      Set_Visibility (Field, Visible);
      Set_Max_Length (Field, Guint16(Maxlength) );

      Pack_End( Hbox, Field, Expand => False );

      PT( Next_Free ).Value_Entry := Field;
      PT( Next_Free ).Description := To_Unbounded_String( Description );

      Pack_Start( Vbox, Hbox, Expand => False );
   end Place;

   -- ********************************************************************** --
   -- ***             P U B L I C    P R O C E D U R E S                 *** --
   -- ********************************************************************** --

   ----------------
   -- Preference --
   ----------------
   function Preference(
      Name : in String ) return String is
      -- return the value of a preference
   begin
      for I in PT'Range loop
         if PT(I).Name /= Null_Unbounded_String
            and then To_String(PT(I).Name)=Name
         then
            return Get_Text( PT(I).Value_Entry );
         end if;
      end loop;

      return "";
   end Preference;

   ---------------------
   -- Add_Preferences --
   ---------------------
   procedure Display(
      This   : in out Object;
      Box    : in out Gtk_Box ) is
      Gbox   : Gtk_Box;
      Parent : Gtk_Box;

      -- add a group box to the parent
      procedure Group(
         Name  : in String;
         Box   : out Gtk_box ) is
         Frame : Gtk_Frame;
      begin
         Gtk_New( Frame );
         Set_Label (Frame, Name );
         Pack_Start( Parent, Frame, Expand => False );

         Gtk_New_Vbox( Box );
         Add(Frame, Box );
         Show_All( Frame );
      end Group;

   begin
      Gtk_New_Vbox (Parent);
      Set_Border_Width (Parent, 5);
      Show_All (Parent);

      ---- AUTHORIZATION --
      Group( "Authorization", Gbox );
      Place(Gbox, Prf_Login   , "Login Name",
                             Maxlength => 20,
                             Default => "gnade" );
      Place(Gbox, Prf_Password, "Password  ",
                             Visible => False,
                             Maxlength => 20,
                             Default => "gnade" );

      ----- DATABASE DEFAULTS ----
      Group( "Database Defaults", Gbox );
      Place(Gbox, Prf_Catalog, "Catalog Name ",
                             Maxlength => 40,
                             Default => "GNADE" );
      Place(Gbox, Prf_Schema, "Schema Name ",
                             Maxlength => 40,
                             Default => "GNADE" );

      ----- PROJECT SETUP -------
      Group( "Project Setup", Gbox );

      Place( Gbox, Prf_Project, "Project Title",
                            Default =>"No Title",
                            Maxlength => 256 );
      Place( Gbox, Prf_Author, "Author",
                            Default =>"No Author",
                            Maxlength => 256 );

      Place( Gbox, Prf_Repository, "Repository Path",
                            Default =>"./",
                            Maxlength => 256 );

      Group( "Program Attributes", Gbox );
      Place( Gbox, Prf_Debug, "Debugging Level",
                            Default =>"0",
                            Maxlength => 256 );

      ----- GSQL Internal  -------
      Group( "GSQL Environment", Gbox );

      Place( Gbox, Prf_Inst_Dir, "GSQL Installation Directory",
                            Default =>"/usr/local/gsql",
                            Maxlength => 256 );

      Pack_Start( Box, Parent );
      Show_All( Box );
   end Display;

   -------------------------
   -- Copy_To_Projectfile --
   -------------------------
   procedure Copy_To_Projectfile is
      -- this procedure copies all preferences into the project file
   begin
      for I in PT'Range loop
         if PT(I).Name /= Null_Unbounded_String then
            Project_File.Update(
               Pref_Prefix, To_String(PT(I).Name),
               Preference( To_String(PT(I).Name) )
            );
         end if;
      end loop;
   end Copy_To_Projectfile;

end GUI_Preferences;

