/*
 *  $Id: fractal.c 22071 2019-05-24 11:52:20Z yeti-dn $
 *  Copyright (C) 2003-2017 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwyresults.h>
#include <libgwyddion/gwyutils.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/fractals.h>
#include <libprocess/stats.h>
#include <libgwydgets/gwygraph.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define FRACTAL_RUN_MODES GWY_RUN_INTERACTIVE

typedef enum {
    GWY_FRACTAL_PARTITIONING  = 0,
    GWY_FRACTAL_CUBECOUNTING  = 1,
    GWY_FRACTAL_TRIANGULATION = 2,
    GWY_FRACTAL_PSDF          = 3,
    GWY_FRACTAL_NMETHODS      = 4
} GwyFractalMethod;

typedef void (*FractalMethodFunc)(GwyDataField *data_field,
                                  GwyDataLine *xresult,
                                  GwyDataLine *yresult,
                                  GwyInterpolationType interpolation);
typedef gdouble (*FractalDimFunc)(GwyDataLine *xresult,
                                  GwyDataLine *yresult,
                                  gdouble *a,
                                  gdouble *b);

typedef struct {
    const gchar *id;
    const gchar *abscissa;
    const gchar *ordinate;
    FractalMethodFunc calculate;
    FractalDimFunc getdim;
} FractalMethod;

typedef struct {
    GwyInterpolationType interp;
    GwyFractalMethod out;
    GwyResultsReportType report_style;
    /* Dynamic state */
    gdouble from[GWY_FRACTAL_NMETHODS];
    gdouble to[GWY_FRACTAL_NMETHODS];
    gdouble result[GWY_FRACTAL_NMETHODS];
    GwyAppDataId target_graph;
} FractalArgs;

typedef struct {
    FractalArgs *args;
    GwyResults *results;
    GwyContainer *data;
    GwyDataField *dfield;
    GtkWidget *from;
    GtkWidget *to;
    GtkWidget *result;
    GtkWidget *interp;
    GtkWidget *out;
    GtkWidget *graph;
    GtkWidget *target_graph;
    GtkWidget *guivalues[GWY_FRACTAL_NMETHODS];
    GtkWidget *rexport;
    GwyGraphModel *gmodel;
} FractalControls;

static gboolean module_register         (void);
static void     fractal                 (GwyContainer *data,
                                         GwyRunType run);
static void     fractal_dialog          (FractalArgs *args,
                                         GwyContainer *data);
static void     interp_changed          (GtkWidget *combo,
                                         FractalArgs *args);
static void     out_changed             (GtkWidget *combo,
                                         FractalControls *controls);
static void     report_style_changed    (FractalControls *controls,
                                         GwyResultsExport *rexport);
static void     fractal_dialog_update   (FractalControls *controls,
                                         FractalArgs *args);
static void     fractal_do              (FractalArgs *args,
                                         FractalControls *controls);
static void     update_graph            (FractalArgs *args,
                                         FractalControls *controls);
static void     fractal_dialog_recompute(FractalControls *controls,
                                         FractalArgs *args);
static void     graph_selected          (GwySelection *selection,
                                         gint hint,
                                         FractalControls *controls);
static void     target_graph_changed    (FractalControls *controls);
static gboolean remove_datapoints       (GwyDataLine *xline,
                                         GwyDataLine *yline,
                                         GwyDataLine *newxline,
                                         GwyDataLine *newyline,
                                         FractalArgs *args);
static void     update_labels           (FractalControls *controls,
                                         FractalArgs *args);
static void     fractal_load_args       (GwyContainer *container,
                                         FractalArgs *args);
static void     fractal_save_args       (GwyContainer *container,
                                         FractalArgs *args);
static void     fractal_sanitize_args   (FractalArgs *args);

static const FractalArgs fractal_defaults = {
    GWY_INTERPOLATION_LINEAR,
    GWY_FRACTAL_PARTITIONING,
    GWY_RESULTS_REPORT_COLON,
    { 0, 0, 0, 0, },
    { 0, 0, 0, 0, },
    { 0, 0, 0, 0, },
    GWY_APP_DATA_ID_NONE,
};

static const FractalMethod methods[] = {
    {
        "partitioning", "log h", "log S",
        gwy_data_field_fractal_partitioning,
        gwy_data_field_fractal_partitioning_dim,
    },
    {
        "cubecounting", "log h", "log N",
        gwy_data_field_fractal_cubecounting,
        gwy_data_field_fractal_cubecounting_dim,
    },
    {
        "triangulation", "log h", "log A",
        gwy_data_field_fractal_triangulation,
        gwy_data_field_fractal_triangulation_dim,
    },
    {
        "psdf", "log k", "log W",
        gwy_data_field_fractal_psdf,
        gwy_data_field_fractal_psdf_dim,
    },
};

static GwyAppDataId target_id = GWY_APP_DATA_ID_NONE;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Calculates fractal dimension using several methods "
       "(partitioning, box counting, triangulation, power spectrum)."),
    "Jindřich Bilek & Petr Klapetek <klapetek@gwyddion.net>",
    "2.1",
    "David Nečas (Yeti) & Petr Klapetek & Jindřich Bílek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, fractal)

static gboolean
module_register(void)
{
    gwy_process_func_register("fractal",
                              (GwyProcessFunc)&fractal,
                              N_("/_Statistics/_Fractal Dimension..."),
                              GWY_STOCK_FRACTAL_MEASURE,
                              FRACTAL_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Calculate fractal dimension"));

    return TRUE;
}

static void
fractal(GwyContainer *data, GwyRunType run)
{
    FractalArgs args;

    g_return_if_fail(run & FRACTAL_RUN_MODES);
    fractal_load_args(gwy_app_settings_get(), &args);
    fractal_dialog(&args, data);
    fractal_save_args(gwy_app_settings_get(), &args);
}

static void
fractal_dialog(FractalArgs *args, GwyContainer *data)
{
    GtkWidget *dialog, *table, *hbox, *vbox;
    GwyGraphArea *area;
    GwySelection *selection;
    FractalControls controls;
    GwyResults *results;
    GwyResultsExport *rexport;
    gint response, row, i, id;
    GString *str;
    GwyEnum *methodsenum;

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &controls.dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(controls.dfield);

    controls.args = args;
    controls.data = data;
    controls.gmodel = gwy_graph_model_new();
    /* The axes should be unitless so no unit initialisation is necessary. */

    controls.results = results = gwy_results_new();
    gwy_results_add_header(results, N_("Fractal Dimension"));
    gwy_results_add_value_str(results, "file", N_("File"));
    gwy_results_add_value_str(results, "image", N_("Image"));
    gwy_results_add_separator(results);

    gwy_results_add_value_plain(results, "partitioning", N_("Partitioning"));
    gwy_results_add_value_plain(results, "cubecounting", N_("Cube counting"));
    gwy_results_add_value_plain(results, "triangulation", N_("Triangulation"));
    gwy_results_add_value_plain(results, "psdf", N_("Power spectrum"));

    gwy_results_fill_filename(results, "file", data);
    gwy_results_fill_channel(results, "image", data, id);

    dialog = gtk_dialog_new_with_buttons(_("Fractal Dimension"), NULL, 0, NULL);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialog),
                                 gwy_stock_like_button_new(_("_Update"),
                                                           GTK_STOCK_EXECUTE),
                                 RESPONSE_CALCULATE);
    gtk_dialog_add_button(GTK_DIALOG(dialog), _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

    vbox = gtk_vbox_new(FALSE, 8);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

    /* Parameters */
    table = gtk_table_new(GWY_FRACTAL_NMETHODS + 8, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);
    row = 0;

    methodsenum = g_new(GwyEnum, G_N_ELEMENTS(methods));
    for (i = 0; i < G_N_ELEMENTS(methods); i++) {
        methodsenum[i].value = i;
        methodsenum[i].name = gwy_results_get_label(results, methods[i].id);
    }
    controls.out = gwy_enum_combo_box_new(methodsenum, G_N_ELEMENTS(methods),
                                          G_CALLBACK(out_changed), &controls,
                                          args->out, FALSE);
    gwy_table_attach_adjbar(table, row, _("_Method:"), NULL,
                            GTK_OBJECT(controls.out),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    controls.interp
        = gwy_enum_combo_box_new(gwy_interpolation_type_get_enum(), -1,
                                 G_CALLBACK(interp_changed), args,
                                 args->interp, TRUE);
    gwy_table_attach_adjbar(table, row, _("_Interpolation type:"), NULL,
                            GTK_OBJECT(controls.interp),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    /* Area */
    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Fit Area")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.from = gtk_label_new(_("minimum"));
    gwy_table_attach_adjbar(table, row, _("From:"), NULL,
                            GTK_OBJECT(controls.from),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    controls.to = gtk_label_new(_("maximum"));
    gwy_table_attach_adjbar(table, row, _("To:"), NULL,
                            GTK_OBJECT(controls.to),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    /* Results */
    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    gtk_table_attach(GTK_TABLE(table), gwy_label_new_header(_("Result")),
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    str = g_string_new(NULL);
    for (i = 0; i < GWY_FRACTAL_NMETHODS; i++) {
        g_string_assign(str, gwy_results_get_label(results, methods[i].id));
        g_string_append_c(str, ':');
        controls.guivalues[i] = gtk_label_new(NULL);
        gwy_table_attach_adjbar(table, row, str->str, NULL,
                                GTK_OBJECT(controls.guivalues[i]),
                                GWY_HSCALE_WIDGET_NO_EXPAND);
        gtk_label_set_selectable(GTK_LABEL(controls.guivalues[i]), TRUE);
        row++;
    }
    g_string_free(str, TRUE);
    row++;

    controls.target_graph = create_target_graph(&args->target_graph,
                                                GTK_WIDGET(table), row++,
                                                controls.gmodel);
    g_signal_connect_swapped(controls.target_graph, "changed",
                             G_CALLBACK(target_graph_changed), &controls);

    controls.rexport = gwy_results_export_new(args->report_style);
    rexport = GWY_RESULTS_EXPORT(controls.rexport);
    gwy_results_export_set_title(rexport, _("Save Fractal Dimension"));
    gwy_results_export_set_results(rexport, controls.results);
    gtk_table_attach(GTK_TABLE(table), controls.rexport,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls.rexport, "format-changed",
                             G_CALLBACK(report_style_changed), &controls);

    /* Graph */
    controls.graph = gwy_graph_new(controls.gmodel);
    g_object_unref(controls.gmodel);
    gtk_widget_set_size_request(controls.graph, 400, 300);

    gtk_box_pack_start(GTK_BOX(hbox), controls.graph, TRUE, TRUE, 4);
    gwy_graph_set_status(GWY_GRAPH(controls.graph), GWY_GRAPH_STATUS_XSEL);
    gwy_graph_enable_user_input(GWY_GRAPH(controls.graph), FALSE);

    area = GWY_GRAPH_AREA(gwy_graph_get_area(GWY_GRAPH(controls.graph)));
    selection = gwy_graph_area_get_selection(area, GWY_GRAPH_STATUS_XSEL);

    gwy_selection_set_max_objects(selection, 1);
    g_signal_connect(selection, "changed",
                     G_CALLBACK(graph_selected), &controls);

    gtk_widget_show_all(dialog);
    fractal_dialog_recompute(&controls, args);

    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            g_free(methodsenum);
            return;
            break;

            case GTK_RESPONSE_OK:
            fractal_do(args, &controls);
            break;

            case RESPONSE_RESET:
            gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls.interp),
                                          args->interp);
            gwy_selection_clear(selection);
            break;

            case RESPONSE_CALCULATE:
            fractal_dialog_recompute(&controls, args);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);
    g_free(methodsenum);
}

static void
interp_changed(GtkWidget *combo, FractalArgs *args)
{
    args->interp = gwy_enum_combo_box_get_active(GTK_COMBO_BOX(combo));
}

/* callback after changed output type */
static void
out_changed(GtkWidget *combo, FractalControls *controls)
{
    GwyGraphArea *area;
    GwySelection *selection;
    FractalArgs *args;

    g_assert(controls->args && controls->data);
    args = controls->args;
    args->out = gwy_enum_combo_box_get_active(GTK_COMBO_BOX(combo));

    gwy_graph_set_status(GWY_GRAPH(controls->graph), GWY_GRAPH_STATUS_XSEL);

    area = GWY_GRAPH_AREA(gwy_graph_get_area(GWY_GRAPH(controls->graph)));
    selection = gwy_graph_area_get_selection(area, GWY_GRAPH_STATUS_XSEL);
    gwy_selection_clear(selection);
    fractal_dialog_update(controls, args);
    update_labels(controls, args);
}

static void
target_graph_changed(FractalControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);

    gwy_data_chooser_get_active_id(chooser, &controls->args->target_graph);
}

static void
report_style_changed(FractalControls *controls, GwyResultsExport *rexport)
{
    controls->args->report_style = gwy_results_export_get_format(rexport);
}

/* update dialog after any recomputation. */
static void
fractal_dialog_update(FractalControls *controls,
                      FractalArgs *args)
{
    update_graph(args, controls);
    if (gwy_results_value_is_na(controls->results, methods[args->out].id))
        gtk_label_set_text(GTK_LABEL(controls->guivalues[args->out]), "");
    else {
        gtk_label_set_text(GTK_LABEL(controls->guivalues[args->out]),
                           gwy_results_get_value(controls->results,
                                                 methods[args->out].id));
    }
}

static void
fractal_do(FractalArgs *args,
           FractalControls *controls)
{
    update_graph(args, controls);
    gwy_app_add_graph_or_curves(controls->gmodel,
                                controls->data, &args->target_graph, 2);
}

static void
update_graph(FractalArgs *args,
             FractalControls *controls)
{
    GwyDataLine *xline, *yline, *xfit, *yfit, *xnline, *ynline;
    GwyGraphCurveModel *gcmodel;
    const gchar *title;
    gint i, res, out;
    gboolean is_line;
    gdouble a, b, dim;
    gdouble *xdata, *ydata;

    out = args->out;
    g_return_if_fail(out < G_N_ELEMENTS(methods));

    xline = gwy_data_line_new(1, 1.0, FALSE);
    yline = gwy_data_line_new(1, 1.0, FALSE);
    xnline = gwy_data_line_new(1, 1.0, FALSE);
    ynline = gwy_data_line_new(1, 1.0, FALSE);

    methods[out].calculate(controls->dfield, xline, yline, args->interp);
    if ((is_line = remove_datapoints(xline, yline, xnline, ynline, args)))
        dim = methods[out].getdim(xnline, ynline, &a, &b);

    gwy_graph_model_remove_all_curves(controls->gmodel);

    gcmodel = gwy_graph_curve_model_new();
    title = gwy_results_get_label(controls->results, methods[out].id);
    g_object_set(gcmodel,
                 "mode", GWY_GRAPH_CURVE_POINTS,
                 "description", title,
                 NULL);
    gwy_graph_curve_model_set_data(gcmodel,
                                   gwy_data_line_get_data_const(xline),
                                   gwy_data_line_get_data_const(yline),
                                   gwy_data_line_get_res(xline));
    g_object_set(controls->gmodel,
                 "title", title,
                 "axis-label-bottom", methods[out].abscissa,
                 "axis-label-left", methods[out].ordinate,
                 NULL);
    gwy_graph_model_add_curve(controls->gmodel, gcmodel);
    g_object_unref(gcmodel);

    res = gwy_data_line_get_res(xnline);
    if (is_line) {
        xfit = gwy_data_line_duplicate(xnline);
        yfit = gwy_data_line_new(res, res, FALSE);
        xdata = gwy_data_line_get_data(xfit);
        ydata = gwy_data_line_get_data(yfit);
        for (i = 0; i < res; i++)
            ydata[i] = xdata[i]*a + b;

        gcmodel = gwy_graph_curve_model_new();
        gwy_graph_curve_model_set_data(gcmodel,
                                       gwy_data_line_get_data_const(xfit),
                                       gwy_data_line_get_data_const(yfit),
                                       gwy_data_line_get_res(xfit));
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "description", _("Linear fit"),
                     NULL);
        gwy_graph_model_add_curve(controls->gmodel, gcmodel);
        g_object_unref(gcmodel);
        g_object_unref(yfit);

        gwy_results_fill_values(controls->results, methods[out].id, dim, NULL);
    }
    else
        gwy_results_set_na(controls->results, methods[out].id, NULL);
    g_object_unref(xline);
    g_object_unref(yline);
    g_object_unref(xnline);
    g_object_unref(ynline);
}

/* (re)compute data and dimension and fits */
static void
fractal_dialog_recompute(FractalControls *controls,
                         FractalArgs *args)
{
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->interp),
                                  args->interp);
    fractal_dialog_update(controls, args);
}

/* update data after selecting something in graph */
static void
graph_selected(GwySelection *selection, gint hint, FractalControls *controls)
{
    FractalArgs *args;
    gint nselections;
    gdouble sel[2];

    g_return_if_fail(hint <= 0);

    nselections = gwy_selection_get_data(selection, NULL);
    gwy_selection_get_object(selection, 0, sel);

    args = controls->args;
    if (nselections == 0 || sel[0] == sel[1]) {
        gtk_label_set_text(GTK_LABEL(controls->from), _("minimum"));
        gtk_label_set_text(GTK_LABEL(controls->to), _("maximum"));
        args->from[args->out] = 0.0;
        args->to[args->out] = 0.0;
    }
    else {
        if (sel[0] > sel[1])
            GWY_SWAP(gdouble, sel[0], sel[1]);
        args->from[args->out] = sel[0];
        args->to[args->out] = sel[1];
        update_labels(controls, args);
    }
}

/* update from and to labels */
static void
update_labels(FractalControls *controls, FractalArgs *args)
{
    gdouble from = 0, to = 0;
    gchar buffer[16];

    from = args->from[args->out];
    to = args->to[args->out];
    if (from == to) {
        gtk_label_set_text(GTK_LABEL(controls->from), _("minimum"));
        gtk_label_set_text(GTK_LABEL(controls->to), _("maximum"));
    }
    else {
        g_snprintf(buffer, sizeof(buffer), "%.2f", from);
        gtk_label_set_text(GTK_LABEL(controls->from), buffer);
        g_snprintf(buffer, sizeof(buffer), "%.2f", to);
        gtk_label_set_text(GTK_LABEL(controls->to), buffer);
    }
}

/* remove datapoints that are below or above selection. New data are in
   newxline and newyline and can be directly used for fitting and fractal
   dimension evaluation. */
static gboolean
remove_datapoints(GwyDataLine *xline, GwyDataLine *yline,
                  GwyDataLine *newxline, GwyDataLine *newyline,
                  FractalArgs *args)
{
    gint i, j, res;
    const gdouble *xdata, *ydata;
    gdouble *newxdata, *newydata;
    gdouble from = 0, to = 0;

    from = args->from[args->out];
    to = args->to[args->out];
    res = gwy_data_line_get_res(xline);
    g_assert(res == gwy_data_line_get_res(yline));
    gwy_data_line_resample(newxline, res, GWY_INTERPOLATION_NONE);
    gwy_data_line_resample(newyline, res, GWY_INTERPOLATION_NONE);

    if (from == to) {
        gwy_data_line_copy(xline, newxline);
        gwy_data_line_copy(yline, newyline);
        return TRUE;
    }

    j = 0;
    xdata = gwy_data_line_get_data_const(xline);
    ydata = gwy_data_line_get_data_const(yline);
    newxdata = gwy_data_line_get_data(newxline);
    newydata = gwy_data_line_get_data(newyline);
    for (i = 0; i < res; i++) {
        if (xdata[i] >= from && xdata[i] <= to) {
            newxdata[j] = xdata[i];
            newydata[j] = ydata[i];
            j++;
        }
    }
    if (j < 2)
        return FALSE;

    gwy_data_line_resize(newxline, 0, j);
    gwy_data_line_resize(newyline, 0, j);

    return TRUE;
}

static const gchar interp_key[]       = "/module/fractal/interp";
static const gchar report_style_key[] = "/module/fractal/report_style";
static const gchar out_key[]          = "/module/fractal/out";

static void
fractal_sanitize_args(FractalArgs *args)
{
    args->interp = gwy_enum_sanitize_value(args->interp,
                                           GWY_TYPE_INTERPOLATION_TYPE);
    args->out = MIN(args->out, GWY_FRACTAL_NMETHODS-1);
    gwy_app_data_id_verify_graph(&args->target_graph);
}

static void
fractal_load_args(GwyContainer *container,
                 FractalArgs *args)
{
    *args = fractal_defaults;

    gwy_container_gis_enum_by_name(container, interp_key, &args->interp);
    gwy_container_gis_enum_by_name(container, report_style_key,
                                   &args->report_style);
    gwy_container_gis_enum_by_name(container, out_key, &args->out);
    args->target_graph = target_id;
    fractal_sanitize_args(args);
}

static void
fractal_save_args(GwyContainer *container,
                 FractalArgs *args)
{
    gwy_container_set_enum_by_name(container, interp_key, args->interp);
    gwy_container_set_enum_by_name(container, report_style_key,
                                   args->report_style);
    gwy_container_set_enum_by_name(container, out_key, args->out);
    target_id = args->target_graph;
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
