/*
 * This file is part of Licq, an instant messaging client for UNIX.
 * Copyright (C) 2010-2012 Licq developers <licq-dev@googlegroups.com>
 *
 * Licq is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Licq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Licq; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef LICQ_PROTOCOLSIGNAL_H
#define LICQ_PROTOCOLSIGNAL_H

#include <list>
#include <pthread.h>
#include <string>

#include "color.h"
#include "userid.h"

namespace Licq
{

class ProtocolSignal
{
public:
  enum SignalType
  {
    SignalLogon         = 1,    // Log on
    SignalLogoff        = 2,    // Log off
    SignalChangeStatus  = 3,    // Change status
    SignalAddUser       = 4,    // Add a contact to list
    SignalRemoveUser    = 5,    // Remove a contact from list
    SignalRenameUser    = 6,    // Rename a contact in server list
    SignalChangeUserGroups=7,   // Change groups for a user
    SignalSendMessage   = 8,    // Send a message to a user
    SignalNotifyTyping  = 9,    // Send typing notification
    SignalGrantAuth     = 10,   // Grant authorization to a user
    SignalRefuseAuth    = 11,   // Refuse authorization for a user
    SignalRequestInfo   = 12,   // Request user information
    SignalUpdateInfo    = 13,   // Update owner information
    SignalRequestPicture= 14,   // Request user picture
    SignalBlockUser     = 15,   // Add user to invisible/block list
    SignalUnblockUser   = 16,   // Remove user from inivisible/block list
    SignalAcceptUser    = 17,   // Add user to visible/accept list
    SignalUnacceptUser  = 18,   // Removed user from visible/accept list
    SignalIgnoreUser    = 19,   // Add user to ignore list
    SignalUnignoreUser  = 20,   // Remove user from ignore list
    SignalSendFile      = 21,   // Send file to user
    SignalSendChat      = 22,   // Send chat request
    SignalCancelEvent   = 23,   // Cancel an event (chat, secure, file, etc.)
    SignalSendReply     = 24,   // Accept/refuse file/chat request
    SignalOpenSecure    = 27,   // Request secure channel with user
    SignalCloseSecure   = 28,   // Close secure channel with user
    SignalRequestAuth   = 29,   // Request authorization from user
    SignalRenameGroup   = 30,   // Rename a user group
    SignalRemoveGroup   = 31,   // Remove a user group
    SignalSendUrl       = 32,   // Send an URL to a user
    SignalProtocolSpecific = 33, // Protocol specific signal
  };

  // Flags for send events
  enum SendFlags
  {
    SendToList          = 1,    // Flag a message as low prio (send to contact list)
    SendUrgent          = 2,    // Flag a message as urgent
    SendDirect          = 4,    // Don't send message via server (if supported by protocol)
    SendToMultiple      = 8,    // Message is sent to multiple recipients
  };

  ProtocolSignal(SignalType signal, const UserId& userId, unsigned long eventId = 0);

  virtual ~ProtocolSignal();

  //! The signal is being sent to the plugin.
  SignalType signal() const
  { return mySignal; }

  //! The user id that this signal is being used for.
  const UserId& userId() const
  { return myUserId; }

  /// Id to use for event generated by this signal
  unsigned long eventId() const
  { return myEventId; }

  //! The calling thread.
  pthread_t callerThread() const
  { return myCallerThread; }


private:
  SignalType mySignal;
  UserId myUserId;
  unsigned long myEventId;
  pthread_t myCallerThread;
};


class ProtoLogonSignal : public ProtocolSignal
{
public:
  ProtoLogonSignal(const UserId& ownerId, unsigned status);
  ~ProtoLogonSignal();

  //! The requested initial status.
  unsigned status() const { return myStatus; }

private:
  unsigned myStatus;
};

class ProtoLogoffSignal : public ProtocolSignal
{
public:
  ProtoLogoffSignal(const UserId& ownerId);
  ~ProtoLogoffSignal();
};

class ProtoChangeStatusSignal : public ProtocolSignal
{
public:
  ProtoChangeStatusSignal(const UserId& ownerId, unsigned status);
  ~ProtoChangeStatusSignal();

  //! The requested status.
  unsigned status() const { return myStatus; }

private:
  unsigned myStatus;
};

class ProtoAddUserSignal : public ProtocolSignal
{
public:
  ProtoAddUserSignal(const UserId& userId);
  ~ProtoAddUserSignal();
};

class ProtoRemoveUserSignal : public ProtocolSignal
{
public:
  ProtoRemoveUserSignal(const UserId& userId);
  ~ProtoRemoveUserSignal();
};

class ProtoRenameUserSignal : public ProtocolSignal
{
public:
  ProtoRenameUserSignal(const UserId& userId);
  ~ProtoRenameUserSignal();
};

class ProtoChangeUserGroupsSignal : public ProtocolSignal
{
public:
  ProtoChangeUserGroupsSignal(const UserId& userId);
  ~ProtoChangeUserGroupsSignal();
};

class ProtoSendMessageSignal : public ProtocolSignal
{
public:
  ProtoSendMessageSignal(unsigned long eventId, const UserId& userId,
      const std::string& message, unsigned flags, const Color* color = NULL,
      unsigned long convoId = 0);
  ~ProtoSendMessageSignal();

  //! The message to be sent
  const std::string& message() const { return myMessage; }
  //! The message flags
  unsigned flags() const { return myFlags; }
  /// Message color
  const Color* color() const { return &myColor; }
  //! The conversation id to use (gets the socket).
  unsigned long convoId() const { return myConvoId; }

private:
  std::string myMessage;
  unsigned myFlags;
  Color myColor;
  unsigned long myConvoId;
};

class ProtoTypingNotificationSignal : public ProtocolSignal
{
public:
  ProtoTypingNotificationSignal(const UserId& userId, bool active, unsigned long convoId = 0);
  ~ProtoTypingNotificationSignal();

  bool active() const { return myActive; }
  //! The conversation id to use (gets the socket).
  unsigned long convoId() const { return myConvoId; }

private:
  bool myActive;
  unsigned long myConvoId;
};

class ProtoGrantAuthSignal : public ProtocolSignal
{
public:
  ProtoGrantAuthSignal(unsigned long eventId, const UserId& userId, const std::string& message);
  ~ProtoGrantAuthSignal();

  const std::string& message() const { return myMessage; }

private:
  std::string myMessage;
};

class ProtoRefuseAuthSignal : public ProtocolSignal
{
public:
  ProtoRefuseAuthSignal(unsigned long eventId, const UserId& userId, const std::string& message);
  ~ProtoRefuseAuthSignal();

  const std::string& message() const { return myMessage; }

private:
  std::string myMessage;
};

class ProtoRequestInfo : public ProtocolSignal
{
public:
  ProtoRequestInfo(unsigned long eventId, const UserId& userId);
  ~ProtoRequestInfo();
};

class ProtoUpdateInfoSignal : public ProtocolSignal
{
public:
  ProtoUpdateInfoSignal(unsigned long eventId, const UserId& ownerId);
  ~ProtoUpdateInfoSignal();
};

class ProtoRequestPicture : public ProtocolSignal
{
public:
  ProtoRequestPicture(unsigned long eventId, const UserId& userId);
  ~ProtoRequestPicture();
};

class ProtoBlockUserSignal : public ProtocolSignal
{
public:
  ProtoBlockUserSignal(const UserId& userId);
  ~ProtoBlockUserSignal();
};

class ProtoUnblockUserSignal : public ProtocolSignal
{
public:
  ProtoUnblockUserSignal(const UserId& userId);
  ~ProtoUnblockUserSignal();
};

class ProtoAcceptUserSignal : public ProtocolSignal
{
public:
  ProtoAcceptUserSignal(const UserId& userId);
  ~ProtoAcceptUserSignal();
};

class ProtoUnacceptUserSignal : public ProtocolSignal
{
public:
  ProtoUnacceptUserSignal(const UserId& userId);
  ~ProtoUnacceptUserSignal();
};

class ProtoIgnoreUserSignal : public ProtocolSignal
{
public:
  ProtoIgnoreUserSignal(const UserId& userId);
  ~ProtoIgnoreUserSignal();
};

class ProtoUnignoreUserSignal : public ProtocolSignal
{
public:
  ProtoUnignoreUserSignal(const UserId& userId);
  ~ProtoUnignoreUserSignal();
};

class ProtoSendFileSignal : public ProtocolSignal
{
public:
  ProtoSendFileSignal(unsigned long eventId, const UserId& userId, const std::string& filename,
      const std::string& message, const std::list<std::string>& files, unsigned flags = 0);
  ~ProtoSendFileSignal();

  const std::string& filename() const { return myFilename; }
  const std::string& message() const { return myMessage; }
  const std::list<std::string>& files() const { return myFiles; }
  unsigned flags() const { return myFlags; }

private:
  std::string myFilename;
  std::string myMessage;
  std::list<std::string> myFiles;
  unsigned myFlags;
};

class ProtoSendChatSignal : public ProtocolSignal
{
public:
  ProtoSendChatSignal(const UserId& userId, const std::string& message);
  ~ProtoSendChatSignal();

  const std::string& message() const { return myMessage; }

private:
  std::string myMessage;
};

class ProtoCancelEventSignal : public ProtocolSignal
{
public:
  ProtoCancelEventSignal(const UserId& userId, unsigned long eventId);
  ~ProtoCancelEventSignal();
};

class ProtoSendEventReplySignal : public ProtocolSignal
{
public:
  ProtoSendEventReplySignal(const UserId& userId, const std::string& message,
      bool accepted, unsigned short port, unsigned long sequence = 0,
      unsigned long flag1 = 0, unsigned long flag2 = 0, bool direct = false,
      const std::string& filename = std::string(), unsigned long filesize = 0);
  ~ProtoSendEventReplySignal();

  const std::string& message() const { return myMessage; }
  bool accept() const { return myAccept; }
  unsigned short port() const { return myPort; }
  unsigned long sequence() const { return mySequence; }
  unsigned long flag1() const { return myFlag1; }
  unsigned long flag2() const { return myFlag2; }
  bool direct() const { return myDirect; }
  const std::string& filename() const { return myFilename; }
  unsigned long filesize() const { return myFilesize; }

private:
  std::string myMessage;
  bool myAccept;
  unsigned short myPort;
  unsigned long mySequence;
  unsigned long myFlag1;
  unsigned long myFlag2;
  bool myDirect;
  std::string myFilename;
  unsigned long myFilesize;
};

class ProtoOpenSecureSignal : public ProtocolSignal
{
public:
  ProtoOpenSecureSignal(unsigned long eventId, const UserId& userId);
  ~ProtoOpenSecureSignal();
};

class ProtoCloseSecureSignal : public ProtocolSignal
{
public:
  ProtoCloseSecureSignal(unsigned long eventId, const UserId& userId);
  ~ProtoCloseSecureSignal();
};

class ProtoRequestAuthSignal : public ProtocolSignal
{
public:
  ProtoRequestAuthSignal(const UserId& userId, const std::string& message);
  ~ProtoRequestAuthSignal();

  const std::string& message() const { return myMessage; }

private:
  std::string myMessage;
};

class ProtoRenameGroupSignal : public ProtocolSignal
{
public:
  ProtoRenameGroupSignal(const UserId& ownerId, int groupId);
  ~ProtoRenameGroupSignal();

  int groupId() const { return myGroupId; }

private:
  int myGroupId;
};

class ProtoRemoveGroupSignal : public ProtocolSignal
{
public:
  ProtoRemoveGroupSignal(const UserId& ownerId, int groupId, unsigned long groupServerId,
      const std::string& groupName);
  ~ProtoRemoveGroupSignal();

  int groupId() const { return myGroupId; }
  unsigned long groupServerId() const { return myGroupServerId; }
  const std::string& groupName() const { return myGroupName; }

private:
  int myGroupId;
  unsigned long myGroupServerId;
  std::string myGroupName;
};

class ProtoSendUrlSignal : public ProtocolSignal
{
public:
  ProtoSendUrlSignal(unsigned long eventId, const UserId& userId, const std::string& url,
      const std::string& message, unsigned flags, const Color* color = NULL);
  ~ProtoSendUrlSignal();

  const std::string& url() const { return myUrl; }
  const std::string& message() const { return myMessage; }
  unsigned flags() const { return myFlags; }
  const Color* color() const { return &myColor; }

private:
  std::string myUrl;
  std::string myMessage;
  unsigned myFlags;
  Color myColor;
};

} // namespace Licq

#endif
