#ifndef _PBUILD_PRIV_H
#define _PBUILD_PRIV_H

/* pdu-priv.h

   Data structures and functions used by all PDU builders.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <dnet.h>
#include <glib.h>

#ifdef HAVE_OPENSSL_MD5_H
#include <openssl/md5.h>
#else
#include "md5.h"
#endif

#include "payload.h"
#include "util.h"
#include "xmalloc.h"
#include "xstrdup.h"
#include "numbers.h"
#include "usr.h"
#include "byteorder.h"
#include "missing.h"
#include "pbuild.h"
#include "builders/pdu-generic.h"


struct pdu_dict {
    char *key;
    void *data;
};

struct fixedp_num {
    uint32_t int_part;
    uint32_t frac_part;
};

/* Default value types */
enum pdu_default_types {
    PDU_DEF_NUM,         /* Default is a simple number                    */
    PDU_DEF_RAND,        /* Default is a random number                    */
    PDU_DEF_PID,         /* Default is PID of current process             */
    PDU_DEF_IP,          /* Default is an IP address                      */
    PDU_DEF_IP6,         /* Default is an IPv6 address                    */
    PDU_DEF_MAC,         /* Default is a MAC address                      */

    PDU_DEF_LEN,         /* Default is the length of a string or raw data */
    PDU_DEF_LEN_PDU,     /* Default is the length of the entire PDU       */
    PDU_DEF_LEN_HDR,     /* Default is length of PDU's header + options   */
    PDU_DEF_LEN_PAYLOAD, /* Default is the length of the PDU's payload    */
    PDU_DEF_LEN_TLV,     /* Default is the length of a type 1 TLV         */
    PDU_DEF_LEN_TLV2,    /* Default is the length of a type 2 TLV         */
    PDU_DEF_LEN_TLV4     /* Default is the length of a type 4 TLV         */
};

/* PDU default value */
struct pdu_defval {
    enum pdu_default_types type;
    union {
	char *__fname;
	ip_addr_t __ipaddr;
	ip6_addr_t __ip6addr;
	uint8_t __macaddr[ETH_ADDR_LEN];
	uint32_t __number;
    } __pdu_defval_u;
};

#define _number __pdu_defval_u.__number
#define _fname __pdu_defval_u.__fname
#define _ipaddr __pdu_defval_u.__ipaddr
#define _ip6addr __pdu_defval_u.__ip6addr
#define _macaddr __pdu_defval_u.__macaddr

typedef struct pdu_defval defval_t;


/* PDU field types */
enum pdu_ftypes {
    /*
     * These do not imply a specific representation on the wire.
     * If the generic pass 1 PDU builder is not being used then
     * these can be used.
     */
    PDU_FTYPE_BRACED_ARGS = 1,
    PDU_FTYPE_NUMTYPE,
    PDU_FTYPE_BOOL,
    PDU_FTYPE_UINT,                 /* General unsigned integer    */

    /*
     * These do imply a specific representation on the wire. If
     * the generic pass 1 PDU builder is being used then these
     * MUST be used.
     */
    PDU_FTYPE_BIT,		    /* Just a bit		    */
    PDU_FTYPE_BITFIELD,		    /* A bit field		    */
    PDU_FTYPE_UINT8,		    /* 8-bit number		    */
    PDU_FTYPE_UINT16,		    /* 16-bit number		    */
    PDU_FTYPE_UINT32,		    /* 32-bit number		    */
    PDU_FTYPE_IP,		    /* An IP address		    */
    PDU_FTYPE_MACADDR,		    /* Ethernet MAC address	    */
    PDU_FTYPE_DATA,		    /* String or raw data	    */
    PDU_FTYPE_IP6ADDR,		    /* An IPv6 address		    */
    PDU_FTYPE_FIXEDP32,		    /* Fixed point number, 16 bits  */
    PDU_FTYPE_FIXEDP64		    /* Fixed point number, 32 bits  */
};

/*
 * Field Type			    Internal Representation
 * --------------------------------------------------------------
 * PDU_FTYPE_NUMTYPE                numspec_t *
 * PDU_FTYPE_BOOL                   pointer to (int) -1 or (int) 0
 * PDU_FTYPE_UINT                   pointer to unsigned int
 * PDU_FTYPE_BIT		    NULL pointer if false, non-NULL otherwise
 * PDU_FTYPE_BITFIELD		    numspec_t *
 * PDU_FTYPE_UINT8		    numspec_t *
 * PDU_FTYPE_UINT16		    numspec_t *
 * PDU_FTYPE_UINT32		    numspec_t *
 * PDU_FTYPE_IP			    numspec_t *
 * PDU_FTYPE_MACADDR		    pointer to MAC address
 * PDU_FTYPE_DATA		    pointer to payload structure
 * PDU_FTYPE_IP6ADDR		    pointer to IPv6 address
 * PDU_FTYPE_FIXEDP16		    pointer to struct fixedp_num
 * PDU_FTYPE_FIXEDP64		    pointer to struct fixedp_num
 */

struct pdu_field {
    char *name;		    /* Must be unique among all fields in a PDU */
    enum pdu_ftypes type;   /* The type of field. Note that the data type
			       on the wire implies internal representation. */
    long offset;	    /* Offset from beginning of the PDU. The units
			       of the offset depend on the type of field.
			       They are normally bytes, but can be bits. */
    size_t size;	    /* Size of field on the wire */
    void *data;		    /* Additional data (depending on field type). */
    defval_t *defval;	    /* Points to default data in some cases */
};

typedef struct pdu_field field_t;


enum node_type {
    PDU_NODE_PDU = 1,
    PDU_NODE_PARM,
    PDU_NODE_BRACED_PARMS
};

struct node_data {
    enum node_type type;
    char *name;
    union {
	struct {
	    const pdu_t *template;
	    size_t hdr_len, opts_len, payload_len;
	    GPtrArray *opts_data;
	} __data_pdu;

	struct {
	    const field_t *field;
	    void *data;
	} __data_parm;

	struct {
	    const field_t *fields;
	} __data_braced;
    } __node_data_u;
};

#define _data_parm __node_data_u.__data_parm
#define _data_pdu __node_data_u.__data_pdu
#define _data_braced __node_data_u.__data_braced


struct _pdu_option {
    char *name;			    /* official, short name of the PDU option
				       (no whitespace!) */
    char *description;		    /* PDU option description */
    size_t len;			    /* Size in bytes of the PDU option */

    /*
     * "Methods"
     */

    void (*build)(const GNode * /* option */, void * /* dest */,
		  void * /* curr_pdu_hdr */, void * /* prev_pdu */);
    size_t (*get_len)(const GNode *);
};

typedef struct _pdu_option pdu_option_t;


/*
 * The PDU structure
 */
struct _pdu {
    char *name;			    /* official, short name of the PDU (no
				       whitespace!) */
    char *description;		    /* PDU description */
    char *documented_in;	    /* where is the PDU documented, e.g RFC */
    size_t len;			    /* Size in bytes of the PDU's header */
    field_t *fields;		    /* Names of the fields in this PDU */
    char *options_class;

    /*
     * "Methods"
     */

    void (*build)(const GNode * /* pdu */, void * /* dest */);
    void (*postbuild)(const GNode * /* pdu */, void * /* dest */,
		      void * /* prev_pdu_hdr */);
    size_t (*get_len)(const GNode *);
};


struct option_data {
    size_t len;
    pdu_option_t *template;
    const GNode *option;
};

extern GHashTable *ethertypes;

extern pdu_t *_pb_getpdutemplate(const char *);
extern pdu_option_t *_pb_getopttemplate(const char *, const char *);
extern const GNode *_pb_findnode(const GNode *, const char *);
extern void _pb_register_protocol_option(const char *, const pdu_option_t *);
extern void _pb_register_protocol(const pdu_t *);
extern GHashTable *_pb_new_hash(const struct pdu_dict *);
extern void *_pb_pdata(const GNode *, const char *);
extern const GNode *_pb_parent_pdu(const GNode *);

/* PDUs */
extern void _pb_register_ipx(void);
extern void _pb_register_ipv4(void);
extern void _pb_register_ipv4_generic(void);
extern void _pb_register_ipv6(void);
extern void _pb_register_icmp(void);
extern void _pb_register_icmp6(void);
extern void _pb_register_igmp(void);
extern void _pb_register_arp(void);
extern void _pb_register_mpls(void);
extern void _pb_register_ether(void);
extern void _pb_register_gre(void);
extern void _pb_register_dot1q(void);
extern void _pb_register_tcp(void);
extern void _pb_register_udp(void);
extern void _pb_register_ospf(void);
extern void _pb_register_bgp(void);
extern void _pb_register_dns(void);
extern void _pb_register_sntp(void);
extern void _pb_register_data(void);
extern void _pb_register_vtp(void);
extern void _pb_register_dtp(void);

#endif /* _PBUILD_PRIV_H */
