/*	Authentication

PIRL CVS ID: Authentication.java,v 1.7 2012/04/16 06:18:23 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Utilities;

import	java.security.SecureRandom;
import	java.security.KeyPair;
import	java.security.PublicKey;
import	java.security.KeyPairGenerator;
import	java.security.InvalidKeyException;
import	java.security.NoSuchAlgorithmException;
import	java.security.NoSuchProviderException;

import	javax.crypto.BadPaddingException;
import	javax.crypto.Cipher;
import	javax.crypto.IllegalBlockSizeException;
import	javax.crypto.NoSuchPaddingException;

import	java.io.ByteArrayOutputStream;
import	java.io.ByteArrayInputStream;
import	java.io.IOException;
import	java.io.ObjectInputStream;
import	java.io.ObjectOutputStream;


/**	The <i>Authentication</i> class contains a set of functions used for
	public-private key pair authentication.
<p>
	The typical uses for the Authentication functions is with a server that
	needs to authenticate a client connection. For example:
<p>
<ol>
<li>Password - The client and the server obtain the same password.
	<p>
	The password, a text string, need not be limited to a single, short
	word; a pass phrase is likely to be more secure. How the client and
	server obtain the password is application specific. A client is
	likely to obtain the password by means of an interactive dialog with
	the user. A server is likely to obtain the password from a
	permission protected file.
	<p>
<li>Keys - The server initializes the public-private key pair.
	<p>
	The key pair provides a public key that will be sent to the client
	and a private key that will be used to decrypt the encrypted
	password returned by the client. This may be done once at server
	startup if a single key pair is considered suffient for
	authenticating all clients. Alternatively each time a client
	connection occurs a key pair may be generated in which case each key
	pair must be associated with the corresponding client.
	<p>
<li>Public_Key - The server generates an encoded public key.
	<p>
	The public key from the key pair is serialized and encoded as a
	hexadecimal representation string. This is done to ensure that the
	public key can be sent by the server to the client even when binary
	data transport is not supported. As with the key pair, this may be
	done once for use with all clients or for each client connection.
	<p>
<li>The server sends the encoded public key to the client.
	<p>
	When a client connects to the server a handshake occurs during which
	authentication information is exchanged. The first step of the
	authentication handshake exchange is the server sending the public
	key to the client.
	<p>
<li>Encoded_Password - The client encryptes and encodes its password
	using the public key.
	<p>
	The public key received from the server is used to encrypt and
	encode the password. As with the public key, the encoded form of
	the encrypted password provides a hexidecimal respresentation
	string of encryption bytes which can be sent to the server even if
	binary data transport is not avaialable.
	<p>
<li>The client sends the encoded password to the server.
	<p>
	This is the second step of the client-server authentication
	handshake exchange.
	<p>
<li>Authenticate - The server decodes and decryptes the encoded password
	using the private key.
	<p>
	The server decodes and decrypts the encoded password received from the
	client using the private key of the key pair. If the resultant
	string matches the password that server knows then the client is
	successfully authenticated. Otherwise the client has failed to
	provide the required authentication information.
</ol>
<p>
	@author		Andrew Davidson and Bradford Castalia - UA/PIRL
	@version	1.7
*/
public class Authentication
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Utilities.Authentication (1.7 2012/04/16 06:18:23)";

private static final int 
	ENCRYPTION_KEY_BYTES	= 512;

private static final String
	ENCRYPTION_METHOD		= "RSA",
	RANDOM_METHOD			= "SHA1PRNG",
	RANDOM_PROVIDER			= "SUN";


//	No constructors for this class.
private Authentication ()
{}

/**	Generate a public-private key pair.
<p>
	A KeyPairGenerator is initialized with a SecureRandom seed and then
	used to generate a KeyPair.
<p>
	@return A KeyPair to be used for asymmetric encryption.
*/
public static KeyPair Keys ()
	throws 
		NoSuchAlgorithmException,
		NoSuchProviderException
{
KeyPairGenerator
	key_pair_generator = KeyPairGenerator.getInstance (ENCRYPTION_METHOD);
key_pair_generator.initialize (ENCRYPTION_KEY_BYTES,
	SecureRandom.getInstance (RANDOM_METHOD, RANDOM_PROVIDER));
return key_pair_generator.generateKeyPair ();
}

/** Create a string representation of a public key.
<p>
	The {@link KeyPair#getPublic() public key} of the {@link #Keys()
	public-private key pair} is serialized as a byte array that is
	{@link #Encode_Hex(byte[]) encoded} as a string in hexadecimal
	notation.
<p>
	An encoded public key is expected to be sent to a client for
	use in generating an {@link #Encoded_Password(String, String)
	encoded password}.
<p>
	@param	keys	A KeyPair that contains a public key.
	@return	A String representation of a public key. This will be null
		if the keys are null.
*/
public static String Public_Key
	(
	KeyPair keys
	)
	throws IOException
{
if (keys == null)
	return null;

ByteArrayOutputStream
	byte_writer = new ByteArrayOutputStream ();
ObjectOutputStream
	serializer = new ObjectOutputStream (byte_writer);
serializer.writeObject (keys.getPublic ());
serializer.flush ();
serializer.close ();

return Encode_Hex (byte_writer.toByteArray ());
}

/** Encrypte and encode a password using an encoded public key.
<p>
	An {@link #Public_Key(KeyPair) encoded public key} is {@link
	#Decode_Hex(String) decoded} from its hexadecimal representation
	and then de-serialzed to a PublicKey object. This is used to
	encrypt the password string. The encrypted bytes are
	{@link #Encode_Hex(byte[]) encoded} into a hexadecimal
	representation string.
<p>
	An encoded password provides secure authentication credentials for a
	client. It is expected to be sent to the server that provided the
	encoded public key for {@link #Authenticate(KeyPair, String, String)
	authentication}.
<p>
	@param	password	The clear text String that is to be encrypted.
	@param 	public_key	A hexadecimal String representation of a
		serialized PublicKey.
	@return	A hexadecimal String representation of the encryped password.
		This will be null if the password is null, the public_key is
		null, the public_key could not be decoded or de-serialized, or
		the password could not be encrypted. Only a valid non-null String
		will be returned.
*/
public static String Encoded_Password
	(
	String password,
	String public_key
	)
{
if (password == null ||
	public_key == null)
	return null;

//	De-serialize the public key.
PublicKey
	key = null;
try {key = (PublicKey)
		(new ObjectInputStream
			(new ByteArrayInputStream
				(Decode_Hex (public_key)))
		.readObject ());}
catch (NumberFormatException e) {}
catch (IOException e) {}
catch (ClassNotFoundException e) {/* Shouldn't happen; PublicKey is a JFC */}
if (key == null)
	return null;

//	Encrypt and encode the password.
try
	{
	Cipher
		encrypter = Cipher.getInstance (key.getAlgorithm ());
	encrypter.init (Cipher.ENCRYPT_MODE, key);
	return Encode_Hex (encrypter.doFinal (password.getBytes ()));
	}
catch (InvalidKeyException e) {}
catch (IllegalBlockSizeException e) {}
catch (NoSuchAlgorithmException e) {}
catch (BadPaddingException e) {}
catch (NoSuchPaddingException e) {}
return null;
}

/** Authenticate an encoded password.
<p>
	The {@link #Encoded_Password(String, String) encoded} password
	string is {@link #Decode_Hex(String) decoded} and then decrypted
	using the {@link KeyPair#getPrivate() private key} of the {@link
	#Keys() key pair}. If the result matches all the same characters of
	the specified password string then authentication has succeeded.
<p>
	<b>N.B.</b>: No exceptions are thrown; if any problem occurs the
	authentication fails.
<p>
	@param	keys	The KeyPair used to decrypt the encoded password. If
		null false is returned.
	@param 	encoded_password	An {@link #Encoded_Password(String, String)
		encoded} password string. If null false is returned.
	@param	password The password string to be compared against the
		encoded password after it has been decoded and decrypted. If
		null false is returned.
	@return true if the decoded encoded password matches the password;
		false otherwise. <b>N.B.</b>: false will be returned if any of
		the arguments are null; any special rules, such as "anonymous"
		authenticaton when the password is null, are the responsibility
		of the application.
*/
public static boolean Authenticate
	(
	KeyPair	keys,
	String	encoded_password,
	String	password
	)
{
if (keys == null ||
	encoded_password == null ||
	password == null)
	return false;

//	Compare the decoded/decrypted encoded password with the expected password.
try
	{
	Cipher
		decryptor = Cipher.getInstance (keys.getPrivate ().getAlgorithm ());
	decryptor.init (Cipher.DECRYPT_MODE, keys.getPrivate ());
	return password.equals
		(new String (decryptor.doFinal (Decode_Hex (encoded_password))));
	}
catch (Exception e) {}
return false;
}

/*==============================================================================
	Utilities
*/
/**	Encode a byte array to a hexadecimal string representation.
<p>
	Each byte in the array is represented by two characters that are
	the hexadecimal value of the byte ('0' padded if the value of
	the byte is less than or equal to 0xF). All hex characters for
	each byte of the array are concatenated in byte array order.
<p>
	@param	bytes	An array of byte values. If null, null is returned.
	@return	A String providing the hexadecimal representation of the
		byte array values.
	@see	#Decode_Hex(String)
*/
public static String Encode_Hex
	(
	byte[]	bytes
	)
{
if (bytes == null)
	return null;

String
	string = "";
int
	value;
for (int
		index = 0;
		index < bytes.length;
		index++)
	{
	value = (int)bytes[index] & 0xFF;
	if (value <= 0xF)
		string += "0";
	string += Integer.toHexString (value).toUpperCase ();
	}
return string;
}

/**	Decode a hexadecimal string to its byte array equivalent.
<p>
	Each pair of characters in the string are translated into the
	binary value they represent and stored in a byte array in the
	order in which they occur in the string.
<p>
	@param	string	A String of hexadecimal character pairs. If
		null, null is returned.
	@return	An array of bytes containing the values represented by
		each character pair in the string in the order they occur.
	@throws	NumberFormatException	If the length of the string is
		odd or any character in it does not represent a hexadecimal
		value (0-9 and a-f, case insensitive).
	@see	#Encode_Hex(byte[])
*/
public static byte[] Decode_Hex
	(
	String	string
	)
{
if (string == null)
	return null;
if ((string.length () % 2) != 0)
	throw new NumberFormatException
		("Invalid odd length hex string to decode -\n"
		+ string);

byte[]
	bytes = new byte[string.length () >> 1];
char
	character;
for (int
		index = 0,
		count = 0;
		count < bytes.length;)
	{
	try
		{
		bytes[count]    = (byte)(Byte_Value (string.charAt (index++)) << 4);
		bytes[count++] += (byte)(Byte_Value (string.charAt (index++)));
		}
	catch (NumberFormatException exception)
		{
		throw new NumberFormatException
			(exception.getMessage ()
			+ string);
		}
	}
return bytes;
}


private static int Byte_Value
	(
	char	character
	)
{
int
	value;
if (character >= '0' &&
	character <= '9')
	return character - '0';
if (character >= 'a' &&
	character <= 'f')
	return character - 'W';
if (character >= 'A' &&
	character <= 'F')
	return character - '7';
throw new NumberFormatException
	("Invalid character '" + character + "' in hex string to decode -\n");
}


}
