/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2012 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#ifndef ANIMATIONTHROW_H
#define ANIMATIONTHROW_H

// C++
#include <cmath>

// Qt
#include <QAbstractAnimation>
#include <QVariant>

// KDE
#include <KDebug>

namespace SmoothTasks {

template <int SIZE, class CLASS>
class AnimationThrow : public QAbstractAnimation
{
// The moc cannot handle template classes, so the propertys may not work,
// see http://doc.qt.nokia.com/qq/qq15-academic.html
// 	Q_OBJECT
	Q_PROPERTY(qreal      acceleration READ acceleration WRITE setAcceleration)
	Q_PROPERTY(CLASS      startValue   READ startValue   WRITE setStartValue  )
	Q_PROPERTY(CLASS      endValue     READ endValue     WRITE setEndValue    )
	Q_PROPERTY(QObject*   targetObject READ targetObject WRITE setTargetObject)
	Q_PROPERTY(QByteArray propertyName READ propertyName WRITE setPropertyName)

public:
	AnimationThrow(QObject* parent = 0);
	AnimationThrow(QObject* target, const QByteArray& propertyName, QObject* parent = 0);
	virtual ~AnimationThrow() {}
	void init();

	/* implement */ virtual int duration() const { return -1; }

	// Getter
	const QObject* targetObject () const { return m_target             ; }
	QObject*       targetObject ()       { return m_target             ; }
	QByteArray     propertyName () const { return m_targetPropertyName ; }
	qreal          acceleration () const { return m_acceleration       ; }
	CLASS          startValue   () const { return convert(m_startValue); }
	CLASS          endValue     () const { return convert(m_endValue  ); }

	// Setter
	void setTargetObject(QObject* target    ) { m_target = target; }
	void setPropertyName(const QByteArray& propertyName) { m_targetPropertyName = propertyName; }
	void setAcceleration(qreal acceleration ) { m_cacheDirty = true; m_acceleration = acceleration; }
	void setStartValue  (const CLASS& value ) { m_cacheDirty = true; convert(value, m_startValue) ; }
	void setEndValue    (const CLASS& value ) { m_cacheDirty = true; convert(value, m_endValue  ) ; }

protected:
	virtual CLASS convert(const qreal src[]) const = 0;
	virtual void  convert(const CLASS& src, qreal dst[]) const = 0;

	/* implement */ virtual void updateCurrentTime(int currentTime);

private:
	void updateCache();

	QObject* m_target;
	QByteArray m_targetPropertyName;
	qreal m_acceleration;
	qreal m_startValue[SIZE];
	qreal m_endValue[SIZE];

	bool  m_cacheDirty;
	int   m_cachedDuration;
	qreal m_cachedDurations[SIZE];
};


template <int SIZE, class CLASS>
AnimationThrow<SIZE, CLASS>::AnimationThrow(QObject* parent)
	: QAbstractAnimation(parent)
	, m_target(0)
	, m_acceleration(1.0)
	, m_cacheDirty(true)
{
	init();
}

template <int SIZE, class CLASS>
AnimationThrow<SIZE, CLASS>::AnimationThrow(QObject* target, const QByteArray& propertyName, QObject* parent)
	: QAbstractAnimation(parent)
	, m_target(target)
	, m_targetPropertyName(propertyName)
	, m_acceleration(1.0)
	, m_cacheDirty(true)
{
	init();
}

template <int SIZE, class CLASS>
void AnimationThrow<SIZE, CLASS>::init()
{
	for (int i = 0; i < SIZE; ++i) {
		m_startValue[i] = 0;
		m_endValue[i] = 0;
	}
}

template <int SIZE, class CLASS>
void AnimationThrow<SIZE, CLASS>::updateCurrentTime(int currentTime)
{
	if (!m_target || m_targetPropertyName.isEmpty()) {
		kWarning() << "target or property name not set";
		stop();
		return;
	}

	updateCache();

	qreal current[SIZE];
	qreal t = 0.001 * static_cast<qreal>(currentTime);
	for (int i = 0; i < SIZE; ++i) {
		if (t < m_cachedDurations[i]) {
			qreal t2 = 1 - t / m_cachedDurations[i];
			current[i] = m_startValue[i] + (1 - t2 * t2) * (m_endValue[i] - m_startValue[i]);
		} else {
			current[i] = m_endValue[i];
		}
	}

	if (!m_target->setProperty(m_targetPropertyName.constData(), QVariant(convert(current)))) {
		kWarning()
			<< "you're trying to animate a non-existing or wrong-typed property"
			<< m_targetPropertyName.constData()
			<< "of your QObject"
			<< m_target;
		stop();
		return;
	}

	if (currentTime >= m_cachedDuration)
		stop();
}

template <int SIZE, class CLASS>
void AnimationThrow<SIZE, CLASS>::updateCache() {
	if (!m_cacheDirty) return;
	qreal dur = 0.0;
	for (int i = 0; i < SIZE; ++i) {
		qreal delta = m_endValue[i] - m_startValue[i];
		m_cachedDurations[i] =  std::sqrt(std::abs(delta) / m_acceleration);
		if (m_cachedDurations[i] > dur)
			dur = m_cachedDurations[i];
	}
	m_cachedDuration = static_cast<int>(1000 * dur) + 1;
	m_cacheDirty = false;
}

} // namespace SmoothTasks

#endif // ANIMATIONTHROW_H
